/*************************************************************************
 * Twinkle Twinkle Little Star on XLOTMIDI – 兼容 tone() 曲谱
 * 硬件：XLOT_Hermes + XLOTMIDI
 *************************************************************************/

#include "XLOTMIDI.h"
#include <math.h>

XLOT_MIDI synth;

/* ─────────── 可调参数 ─────────── */
#define INSTRUMENT_PROGRAM  GM_MUSIC_BOX   // 音乐盒，很适合小星星
#define TEMPO               100            // BPM
#define CHANNEL             0              // MIDI 通道 0–15
#define VELOCITY            110            // 力度 0–127
#define UART_TX_PIN         18                    // 主控 TX → XLOTMIDI RX  (绿色线/Green Wire)XLOT-Hermes→18 XLOT-Nova→4 XLOT-MiniC6→2 
#define UART_RX_PIN         17                    // 主控 RX → XLOTMIDI TX  （黄色线/Yellow Wire）XLOT-Hermes→17 XLOT-Nova→1 XLOT-MiniC6→1
#define LOOP_GAP_MS         500            // 每遍曲子的停顿间隔
/* ───────────────────────────────── */

/*********************************************************
 * General MIDI Level 1 — 128 Instrument Definitions
 * GM 标准乐器 128 个（全部带中英文注释）
 *********************************************************/

/* 0–7: 钢琴类 Piano */
#define GM_ACOUSTIC_GRAND_PIANO      0   // Acoustic Grand Piano  大钢琴（最经典的钢琴音色）
#define GM_BRIGHT_ACOUSTIC_PIANO     1   // Bright Piano         明亮钢琴（更清脆）
#define GM_ELECTRIC_GRAND_PIANO      2   // Electric Grand       电钢琴（老式）
#define GM_HONKY_TONK_PIANO          3   // Honky-tonk           酒吧琴（走音效果）
#define GM_ELECTRIC_PIANO_1          4   // Electric Piano 1     电钢琴1（柔和）
#define GM_ELECTRIC_PIANO_2          5   // Electric Piano 2     电钢琴2（加合唱效果）
#define GM_HARPSICHORD               6   // Harpsichord          羽管键琴（巴洛克）
#define GM_CLAVI                     7   // Clavinet             电击琴（FUN K音乐常用）

/* 8–15: 色击乐器 Chromatic Percussion */
#define GM_CELESTA                   8   // Celesta              钢片琴（童话感）
#define GM_GLOCKENSPIEL              9   // Glockenspiel         钟琴（清脆金属）
#define GM_MUSIC_BOX                10   // Music Box            八音盒（治愈系）
#define GM_VIBRAPHONE               11   // Vibraphone           颤音琴（柔金属）
#define GM_MARIMBA                  12   // Marimba              马林巴（木音块）
#define GM_XYLOPHONE                13   // Xylophone            木琴（高频亮）
#define GM_TUBULAR_BELLS            14   // Tubular Bells        管钟（宗教感）
#define GM_DULCIMER                 15   // Dulcimer             扬琴（古典民族味）

/* 16–23: 风琴 Organ */
#define GM_DRAWBAR_ORGAN            16   // Drawbar Organ        拉杆风琴（爵士）
#define GM_PERCUSSIVE_ORGAN         17   // Percussive Organ     击杆风琴
#define GM_ROCK_ORGAN               18   // Rock Organ           摇滚风琴
#define GM_CHURCH_ORGAN             19   // Church Organ         教堂风琴（宏大）
#define GM_REED_ORGAN               20   // Reed Organ           簧风琴
#define GM_ACCORDION                21   // Accordion            手风琴
#define GM_HARMONICA                22   // Harmonica            口琴
#define GM_TANGO_ACCORDION          23   // Tango Accordion      探戈手风琴（阿根廷味）

/* 24–31: 吉他 Guitar */
#define GM_NYLON_ACOUSTIC_GUITAR    24   // Nylon Guitar         尼龙吉他（古典）
#define GM_STEEL_ACOUSTIC_GUITAR    25   // Steel Guitar         钢弦吉他（流行）
#define GM_ELECTRIC_JAZZ_GUITAR     26   // Jazz Guitar          爵士电吉他
#define GM_ELECTRIC_CLEAN_GUITAR    27   // Clean Guitar         清音电吉他
#define GM_ELECTRIC_MUTED_GUITAR    28   // Muted Guitar         闷音电吉他
#define GM_OVERdrive_GUITAR         29   // Overdrive Guitar     过载电吉他
#define GM_DISTORTION_GUITAR        30   // Distortion Guitar    失真吉他（摇滚必备）
#define GM_GUITAR_HARMONICS         31   // Guitar Harmonics     吉他泛音

/* 32–39: 贝斯 Bass */
#define GM_ACOUSTIC_BASS            32   // Acoustic Bass        原声贝斯
#define GM_ELECTRIC_BASS_FINGER     33   // Finger Bass          电贝司（手指弹）
#define GM_ELECTRIC_BASS_PICK       34   // Pick Bass            拨片贝斯
#define GM_FRETLESS_BASS            35   // Fretless Bass        无品贝斯
#define GM_SLAP_BASS_1              36   // Slap Bass 1          拍弦贝斯1
#define GM_SLAP_BASS_2              37   // Slap Bass 2          拍弦贝斯2
#define GM_SYNTH_BASS_1             38   // Synth Bass 1         合成贝斯1
#define GM_SYNTH_BASS_2             39   // Synth Bass 2         合成贝斯2（重）

/* 40–47: 弦乐 Strings */
#define GM_VIOLIN                   40   // Violin               小提琴
#define GM_VIOLA                    41   // Viola                中提琴
#define GM_CELLO                    42   // Cello                大提琴
#define GM_CONTRABASS               43   // Contrabass           低音提琴
#define GM_TREMOLO_STRINGS          44   // Tremolo Strings      颤音弦乐
#define GM_PIZZICATO_STRINGS        45   // Pizzicato Strings    拨弦弦乐
#define GM_ORCHESTRAL_HARP          46   // Harp                 竖琴
#define GM_TIMPANI                  47   // Timpani              定音鼓

/* 48–55: 弦乐合奏 Ensemble */
#define GM_STRING_ENSEMBLE_1        48   // Strings 1            弦乐合奏1
#define GM_STRING_ENSEMBLE_2        49   // Strings 2            弦乐合奏2
#define GM_SYNTH_STRINGS_1          50   // Synth Strings 1      合成弦乐1
#define GM_SYNTH_STRINGS_2          51   // Synth Strings 2      合成弦乐2
#define GM_CHOIR_AAHS               52   // Choir Aahs           人声“啊”
#define GM_VOICE_OOHS               53   // Voice Oohs           人声“喔”
#define GM_SYNTH_CHOIR              54   // Synth Choir          合成唱诗
#define GM_ORCHESTRA_HIT            55   // Orchestra Hit        管弦敲击（特效）

/* 56–63: 铜管 Brass */
#define GM_TRUMPET                  56   // Trumpet              小号
#define GM_TROMBONE                 57   // Trombone             长号
#define GM_TUBA                     58   // Tuba                 大号
#define GM_MUTED_TRUMPET            59   // Muted Trumpet        弱音小号
#define GM_FRENCH_HORN              60   // French Horn          圆号
#define GM_BRASS_SECTION            61   // Brass Section        铜管合奏
#define GM_SYNTH_BRASS_1            62   // Synth Brass 1        合成铜管1
#define GM_SYNTH_BRASS_2            63   // Synth Brass 2        合成铜管2

/* 64–71: 簧管 Reed */
#define GM_SOPRANO_SAX              64   // Soprano Sax          高音萨克斯
#define GM_ALTO_SAX                 65   // Alto Sax             中音萨克斯
#define GM_TENOR_SAX                66   // Tenor Sax            次中音萨克斯
#define GM_BARITONE_SAX             67   // Baritone Sax         上低音萨克斯
#define GM_OBOE                     68   // Oboe                 双簧管
#define GM_ENGLISH_HORN             69   // English Horn         英国管
#define GM_BASSOON                  70   // Bassoon              巴松
#define GM_CLARINET                 71   // Clarinet             单簧管（黑管）

/* 72–79: 长笛/吹管 Flute / Pipe */
#define GM_PICCOLO                  72   // Piccolo              短笛
#define GM_FLUTE                    73   // Flute                长笛
#define GM_RECORDER                 74   // Recorder             竖笛
#define GM_PAN_FLUTE                75   // Pan Flute            排箫
#define GM_BLOWN_BOTTLE             76   // Blown Bottle         吹瓶声
#define GM_SHAKUHACHI               77   // Shakuhachi           尺八
#define GM_WHISTLE                  78   // Whistle              哨子
#define GM_OCARINA                  79   // Ocarina              陶笛

/* 80–87: 主音 Lead */
#define GM_SYNTH_LEAD_SQUARE        80   // Lead Square          方波主音
#define GM_SYNTH_LEAD_SAWTOOTH      81   // Lead Saw             锯齿波主音
#define GM_SYNTH_LEAD_CALLIOPE      82   // Calliope             风琴主音
#define GM_SYNTH_LEAD_CHIFF         83   // Chiff                吹奏主音
#define GM_SYNTH_LEAD_CHARANG       84   // Charang              吉他合成主音
#define GM_SYNTH_LEAD_VOICE         85   // Voice Lead           人声主音
#define GM_SYNTH_LEAD_FIFTHS        86   // Fifths               五度合成主音
#define GM_SYNTH_LEAD_BASS_LEAD     87   // Bass & Lead          低音 + 主音

/* 88–95: 垫底 Pad */
#define GM_SYNTH_PAD_NEW_AGE        88   // Pad New Age          氛围垫底
#define GM_SYNTH_PAD_WARM           89   // Pad Warm             温暖垫底
#define GM_SYNTH_PAD_POLYSYNTH      90   // Pad Polysynth        合成铺底
#define GM_SYNTH_PAD_CHOIR          91   // Pad Choir            合唱铺底
#define GM_SYNTH_PAD_BOWED          92   // Pad Bowed            弓形垫底
#define GM_SYNTH_PAD_METALLIC       93   // Pad Metallic         金属垫底
#define GM_SYNTH_PAD_HALo           94   // Pad Halo             光晕垫底
#define GM_SYNTH_PAD_SWEEP          95   // Pad Sweep            扫弦垫底

/* 96–103: 特效 FX */
#define GM_SYNTH_FX_RAIN            96   // FX Rain              雨声
#define GM_SYNTH_FX_SOUNDTRACK      97   // FX Soundtrack        电影音效
#define GM_SYNTH_FX_CRYSTAL         98   // FX Crystal           水晶
#define GM_SYNTH_FX_ATMOSPHERE      99   // FX Atmosphere        氛围层
#define GM_SYNTH_FX_BRIGHTNESS     100   // FX Brightness        明亮效果
#define GM_SYNTH_FX_GOBLINS        101   // FX Goblins           小妖怪
#define GM_SYNTH_FX_ECHOES         102   // FX Echoes            回声
#define GM_SYNTH_FX_SCI_FI         103   // FX Sci-Fi            科幻效果

/* 104–111: 民族乐器 Ethnic */
#define GM_SITAR                   104   // Sitar                西塔琴（印度）
#define GM_BANJO                   105   // Banjo                班卓琴
#define GM_SHAMISEN                106   // Shamisen             三味线
#define GM_KOTO                    107   // Koto                 和琴
#define GM_KALIMBA                 108   // Kalimba              拇指琴
#define GM_BAGPIPE                 109   // Bagpipe              风笛
#define GM_FIDDLE                  110   // Fiddle               民间小提琴
#define GM_SHANAI                  111   // Shanai               绍奈（类似唢呐）

/* 112–119: 打击 Percussion */
#define GM_TINKLE_BELL             112   // Tinkle Bell          铃铛
#define GM_AGOGO                   113   // Agogo                木鱼
#define GM_STEEL_DRUMS             114   // Steel Drums          钢鼓
#define GM_WOODBLOCK               115   // Woodblock            木块
#define GM_TAIKO_DRUM              116   // Taiko Drum           太鼓
#define GM_MELODIC_TOM             117   // Melodic Tom          旋律桶鼓
#define GM_SYNTH_DRUM              118   // Synth Drum           合成鼓
#define GM_REVERSE_CYMBAL          119   // Reverse Cymbal       倒放镲片

/* 120–127: 特效 SFX */
#define GM_GUITAR_FRET_NOISE       120   // Guitar Fret Noise    吉他换把噪声
#define GM_BREATH_NOISE            121   // Breath Noise         呼吸声
#define GM_SEASHORE                122   // Seashore             海浪声
#define GM_BIRD_TWEET              123   // Bird Tweet           鸟叫
#define GM_TELEPHONE_RING          124   // Telephone Ring       电话
#define GM_HELICOPTER              125   // Helicopter           直升机
#define GM_APPLAUSE                126   // Applause             掌声
#define GM_GUNSHOT                 127   // Gunshot              枪声

/*====================================================================*/

/* ───── NOTE Frequencies ───── */
#define NOTE_B0  31
#define NOTE_C1  33
#define NOTE_CS1 35
#define NOTE_D1  37
#define NOTE_DS1 39
#define NOTE_E1  41
#define NOTE_F1  44
#define NOTE_FS1 46
#define NOTE_G1  49
#define NOTE_GS1 52
#define NOTE_A1  55
#define NOTE_AS1 58
#define NOTE_B1  62
#define NOTE_C2  65
#define NOTE_CS2 69
#define NOTE_D2  73
#define NOTE_DS2 78
#define NOTE_E2  82
#define NOTE_F2  87
#define NOTE_FS2 93
#define NOTE_G2  98
#define NOTE_GS2 104
#define NOTE_A2  110
#define NOTE_AS2 117
#define NOTE_B2  123
#define NOTE_C3  131
#define NOTE_CS3 139
#define NOTE_D3  147
#define NOTE_DS3 156
#define NOTE_E3  165
#define NOTE_F3  175
#define NOTE_FS3 185
#define NOTE_G3  196
#define NOTE_GS3 208
#define NOTE_A3  220
#define NOTE_AS3 233
#define NOTE_B3  247
#define NOTE_C4  262
#define NOTE_CS4 277
#define NOTE_D4  294
#define NOTE_DS4 311
#define NOTE_E4  330
#define NOTE_F4  349
#define NOTE_FS4 370
#define NOTE_G4  392
#define NOTE_GS4 415
#define NOTE_A4  440
#define NOTE_AS4 466
#define NOTE_B4  494
#define NOTE_C5  523
#define NOTE_CS5 554
#define NOTE_D5  587
#define NOTE_DS5 622
#define NOTE_E5  659
#define NOTE_F5  698
#define NOTE_FS5 740
#define NOTE_G5  784
#define NOTE_GS5 831
#define NOTE_A5  880
#define NOTE_AS5 932
#define NOTE_B5  988
#define NOTE_C6  1047
#define NOTE_CS6 1109
#define NOTE_D6  1175
#define NOTE_DS6 1245
#define NOTE_E6  1319
#define NOTE_F6  1397
#define NOTE_FS6 1480
#define NOTE_G6  1568
#define NOTE_GS6 1661
#define NOTE_A6  1760
#define NOTE_AS6 1865
#define NOTE_B6  1976
#define NOTE_C7  2093
#define NOTE_CS7 2217
#define NOTE_D7  2349
#define NOTE_DS7 2489
#define NOTE_E7  2637
#define NOTE_F7  2794
#define NOTE_FS7 2960
#define NOTE_G7  3136
#define NOTE_GS7 3322
#define NOTE_A7  3520
#define NOTE_AS7 3729
#define NOTE_B7  3951
#define NOTE_C8  4186
#define NOTE_CS8 4435
#define NOTE_D8  4699
#define NOTE_DS8 4978
#define REST      0

/* ───── 频率 → MIDI ───── */
static int freqToMidi(int f) {
  if (f <= 0) return -1;
  double n = 69.0 + 12.0 * (log((double)f / 440.0) / log(2.0));
  int midi = (int)round(n);
  if (midi < 0) midi = 0;
  if (midi > 127) midi = 127;
  return midi;
}

static inline int toMidiIfNeeded(int v) {
  if (v == REST) return -1;
  if (v <= 127) return v;
  return freqToMidi(v);
}

constexpr int WHOLE_NOTE_MS = (60000 * 4) / TEMPO;

/*=====================================================================
 *  🎵 Twinkle Twinkle Little Star Melody
 *  调：C，大部分四分音符，句尾二分音符拉长
 *====================================================================

Twinkle, twinkle, little star,
How I wonder what you are.
Up above the world so high,
Like a diamond in the sky.
Twinkle, twinkle, little star,
How I wonder what you are.

=====================================================================*/

int melody[] = {
  // Twinkle, twinkle, little star,
  NOTE_C4,4, NOTE_C4,4, NOTE_G4,4, NOTE_G4,4,
  NOTE_A4,4, NOTE_A4,4, NOTE_G4,2,

  // How I wonder what you are.
  NOTE_F4,4, NOTE_F4,4, NOTE_E4,4, NOTE_E4,4,
  NOTE_D4,4, NOTE_D4,4, NOTE_C4,2,

  // Up above the world so high,
  NOTE_G4,4, NOTE_G4,4, NOTE_F4,4, NOTE_F4,4,
  NOTE_E4,4, NOTE_E4,4, NOTE_D4,2,

  // Like a diamond in the sky.
  NOTE_G4,4, NOTE_G4,4, NOTE_F4,4, NOTE_F4,4,
  NOTE_E4,4, NOTE_E4,4, NOTE_D4,2,

  // Twinkle, twinkle, little star,
  NOTE_C4,4, NOTE_C4,4, NOTE_G4,4, NOTE_G4,4,
  NOTE_A4,4, NOTE_A4,4, NOTE_G4,2,

  // How I wonder what you are.
  NOTE_F4,4, NOTE_F4,4, NOTE_E4,4, NOTE_E4,4,
  NOTE_D4,4, NOTE_D4,4, NOTE_C4,2
};

const int MELODY_LEN = sizeof(melody) / sizeof(melody[0]);

/*=====================================================================
 *  播放函数
 *====================================================================*/
void playTuneInterleaved() {
  for (int i = 0; i < MELODY_LEN; i += 2) {

    int noteVal = melody[i];
    int div     = melody[i + 1];

    int noteTime = WHOLE_NOTE_MS / div;

    int midi = toMidiIfNeeded(noteVal);

    if (midi >= 0) {
      synth.setNoteOn(CHANNEL, midi, VELOCITY);
      delay((int)(noteTime * 0.9));
      synth.setNoteOff(CHANNEL, midi, 0);
      delay((int)(noteTime * 0.1));
    } else {
      delay(noteTime);
    }
  }
}

/*=====================================================================
 *  Setup / Loop
 *====================================================================*/
void setup() {
  Serial.begin(115200);
  synth.begin(&Serial1, XLOT_MIDI_BAUD, UART_TX_PIN, UART_RX_PIN);

  synth.setInstrument(0, CHANNEL, INSTRUMENT_PROGRAM);

  Serial.println("✨ Twinkle Twinkle Little Star Ready");
}

void loop() {
  playTuneInterleaved();
  delay(LOOP_GAP_MS);
}
